package gov.va.genisis2.bo;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.model.CommentHistory;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.model.Source;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.model.Users;
import gov.va.genisis2.model.WorkflowStatus;
import gov.va.genisis2.service.impl.CommentHistoryService;
import gov.va.genisis2.service.impl.LookUpService;
import gov.va.genisis2.service.impl.RequestService;
import gov.va.genisis2.service.impl.StudyApprovalService;
import gov.va.genisis2.service.impl.UserManagementService;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;
import gov.va.genisis2.vo.CopyTableDomains;
import gov.va.genisis2.vo.CopyTableSource;
import gov.va.genisis2.vo.EmailDetails;
import gov.va.genisis2.vo.GenisisProperties;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;


/**
 * The Class BusinessService.
 * 
 * The BusinessService Class defines the methods that are called by the
 * Genisis2Controller to perform the business logic.
 * 
 */
/*
 * TBD: Consolidate all genisis2.properties into genisisProperties
 */

public class BusinessService implements IBusiness {

	/** The logger. */
	private final org.slf4j.Logger logger = LoggerFactory.getLogger(BusinessService.class);
	
	/** The request service. */
	private RequestService requestService;

	/** The study approval service. */
	private StudyApprovalService studyApprovalService;

	/** The comment history service. */
	private CommentHistoryService commentHistoryService;

	/** The user management service. */
	private UserManagementService userManagementService;

	/** The activiti service. */
	private ActivitiService activitiService;

	/** The look up service. */
	private LookUpService lookUpService;

	/** The copy table source. */
	private CopyTableSource copyTableSource;
	
	/*
	 * The copy table domain details 
	 */
	private CopyTableDomains copyTableDomains;
	
	/*
	 * The genisisProperties
	 */
	private GenisisProperties genisisProperties;
	

	
	
	
	/**
	 * Creates the study approval.
	 *
	 * @param studyApprovalIn The studyApprovalIn.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  createStudyApproval(StudyApproval studyApprovalIn) {
		logger.info("BusinessService:  createStudyApproval");
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApproval studyApproval = BusinessHelper.handleStudyApprovalPayload(studyApprovalIn);
		int id = getStudyApprovalService().createStudyApproval(studyApproval);
		if (id > 0) {
			wrapper = this.getStudyApprovalsByID(id);
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
		} else {
			List<StudyApproval> studyApprovalList = BusinessHelper
					.prepareStudyApprovalResponsePayload(Arrays.asList(studyApproval));
			wrapper.setResponse(studyApprovalList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Study Approval");
		}
		return wrapper;
	}

	/**
	 * Update study approval.
	 *
	 * @param studyApprovalIn The studyApprovalIn.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  updateStudyApproval(StudyApproval studyApprovalIn) {
		logger.info("BusinessService: updateStudyApproval");
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApproval studyApproval = BusinessHelper.handleStudyApprovalPayload(studyApprovalIn);
		int id = studyApproval.getId();
		if (id > 0) {
			getStudyApprovalService().updateStudyApproval(studyApproval);
			wrapper = this.getStudyApprovalsByID(id);
		} else {
			List<StudyApproval> studyApprovalList = BusinessHelper
					.prepareStudyApprovalResponsePayload(Arrays.asList(studyApproval));
			wrapper.setResponse(studyApprovalList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to update Study Approval");
		}
		return wrapper;
	}

	/**
	 * Gets the study approvals by ID.
	 *
	 * @param id    The id.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getStudyApprovalsByID(int id) {
		logger.info("BusinessService:  getStudyApprovalsByID:" + id);
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApproval studyApproval = getStudyApprovalService().getStudyApprovalsByID(id);
		if (studyApproval != null) {
			List<StudyApproval> studyApprovalList = BusinessHelper
					.prepareStudyApprovalResponsePayload(Arrays.asList(studyApproval));
			wrapper.setResponse(studyApprovalList.get(0));
		}
		return createResponseWrapper(wrapper);	
	}

	/**
	 * Gets the study approvals by UID.
	 *
	 * @param uid The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getStudyApprovalsByUID(String uid) {
		logger.info("BusinessService:  getStudyApprovals by user:" + uid);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<StudyApproval> studyApprovalList = getStudyApprovalService().getStudyApprovalsByUID(uid);
		studyApprovalList = BusinessHelper.prepareStudyApprovalResponsePayload(studyApprovalList);
		wrapper.setResponse(studyApprovalList);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the study approvals.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getStudyApprovals() {
		logger.info("BusinessService:  getStudyApprovals");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<StudyApproval> studyApprovalList = getStudyApprovalService().getStudyApprovals();
		studyApprovalList = BusinessHelper.prepareStudyApprovalResponsePayload(studyApprovalList);
		wrapper.setResponse(studyApprovalList);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the request data sources.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getRequestDataSources() {
		logger.info("BusinessService:  getDataSources");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Source> listDatasources = getRequestService().getDataSources();
		wrapper.setResponse(listDatasources);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the request types.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getRequestTypes() {
		logger.info("BusinessService:  getRequestTypes");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestType> listRequestTypes = getRequestService().getRequestTypes();
		wrapper.setResponse(listRequestTypes);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Creates the response wrapper.
	 *
	 * @param wrapper
	 *            The wrapper.
	 * @return ResponseWrapper This returns wrapper.
	 */
	private ResponseWrapper createResponseWrapper(ResponseWrapper wrapper) {
		logger.info("BusinessService:  creating Response Wrapper");
		if (wrapper == null) {
			ResponseWrapper wrapperLocal = new ResponseWrapper();
			wrapperLocal.setMessage("Internal System Error. Could not perform requested Opeartion");
			wrapperLocal.setResponse(null);
			wrapperLocal.setSuccess(false);
			return wrapperLocal;
		}

		if (wrapper.getResponse() == null) {
			wrapper.setSuccess(true);
			wrapper.setResponse(null);
			wrapper.setMessage(CommonEnum.NO_RECORD_FOUND.getText());
			return wrapper;
		} 
		
		
		Object object = wrapper.getResponse();
		
		String isOk;
		for (;;) {
			if (object instanceof ArrayList<?>
					|| object instanceof StudyApproval
					|| object instanceof CommentHistory
					|| object instanceof Request) {
				isOk = "true";
				
			}
			else {isOk = "false";}
			break;
		}
				
		if (isOk=="false") {
			wrapper.setSuccess(true);
			wrapper.setResponse(null);
			wrapper.setMessage(CommonEnum.NO_RECORD_FOUND.getText());
			return wrapper;
		}
			
		if  (!(object instanceof ArrayList<?>)) {
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
			wrapper.setResponse(object);
			return wrapper;
		}
		
		try {
			wrapper.setSuccess(true);
			wrapper.setMessage(null);

			if (((ArrayList) object).size() == 1) {
				Object obj = ((ArrayList<?>) object).get(0);

				if (obj instanceof StudyApproval) {
					StudyApproval studyApproval = (StudyApproval) (obj);
					wrapper.setResponse(studyApproval);
				} else if (obj instanceof CommentHistory) {
					CommentHistory commentHistory = (CommentHistory) (obj);
					wrapper.setResponse(commentHistory);
				}

				return wrapper;
			}

		} catch (Exception ex) {
			logger.error(
					"Exception occured in the method createResponseWrapper. Reason: "
							+ ex.getMessage() + " ", ex);

		}
		
		return wrapper;
	}

	/**
	 * Get user details by email.
	 *
	 * @param email
	 *            The email.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  getUserDetailsByEmail(String email) {
		logger.info("BusinessService:  user details by email:" + email);
		ResponseWrapper wrapper = new ResponseWrapper();
		Users users = null;

		try {
			users = userManagementService.getUserDetailsByEmail(email);
		} catch (Exception ex) {
			logger.error("Error occured while getting user details on email. " + email + ". Reason: " + ex.getMessage() + " ", ex);
			wrapper.setResponse(users);
			wrapper.setSuccess(false);
			wrapper.setMessage("Error ocuured while getting User details for " + email);
			return wrapper;
		}

		if (users == null) {
			wrapper.setMessage("No records found.");
			wrapper.setSuccess(false);
			wrapper.setResponse(null);
		} else {
			wrapper.setSuccess(true);
			wrapper.setResponse(users);
		}

		return wrapper;
	}

	/**
	 * Get user details by id.
	 *
	 * @param id   The id.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  getUserDetailsById(int id) {
		logger.info("BusinessService:  user details by Id:" + id);
		ResponseWrapper wrapper = new ResponseWrapper();
		Users users = null;

		try {
			users = userManagementService.getUserDetailsById(id);
		} catch (Exception ex) {
			logger.error("Error occured while getting user details on id-'" + id + "'. Reason: " + ex.getMessage() + " ", ex);
			wrapper.setResponse(users);
			wrapper.setSuccess(false);
			wrapper.setMessage("Error ocuured while getting User details for " + id);
			return wrapper;
		}

		if (users == null) {
			wrapper.setMessage("No records found.");
			wrapper.setSuccess(false);
			wrapper.setResponse(null);
		} else {
			wrapper.setResponse(users);
			wrapper.setSuccess(true);
		}

		return wrapper;
	}

	/**
	 * Gets the user role.
	 *
	 * @param uid   The uid.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  getUserRole(int uid) {
		logger.info("BusinessService:  user role by uid:" + uid);
		ResponseWrapper wrapper = new ResponseWrapper();
		String userRole = userManagementService.getUserRole(uid);

		if (StringUtils.isBlank(userRole) ) {
			wrapper.setMessage("No roles found.");
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
		} else {
			wrapper.setResponse(userRole);
			wrapper.setSuccess(true);
		}

		return wrapper;
	}

	/**
	 * Creates the request.
	 *
	 * @param requestIn The requestIn.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  createRequest(Request requestIn) {
		logger.info("BusinessService: createRequest");
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = BusinessHelper.handleRequestPayload(requestIn);

		if (StringUtils.isBlank(request.getApproverId())) {
			Users user = this.userManagementService.getUserDetailsByEmail(request.getCreatedBy());
			if (user != null)
			request.setApproverId(user.getApprover());
		}

		// Implement workflow
		boolean workflowProcessIdRequired = true;
		if (request.getId() != 0) {
			Request existingRequest = getRequestService().getRequestsById(request.getId());
			if (existingRequest.getId() != 0) {
				workflowProcessIdRequired = false;
			}
		}
		if (workflowProcessIdRequired) {
		
			String processInstanceId = StringUtils.EMPTY;

			try {
				processInstanceId = activitiService.startProcess(request.getCreatedBy());
			} catch (Exception e) {
				logger.error("Unable to start process for " + request.getCreatedBy() + ". Reason: " + e.getMessage() + " ", e);
			}

			if (StringUtils.isBlank(processInstanceId)) {
				request.setProcessId(0);
			} else {
				int intProcessInstanceId = Integer.parseInt(processInstanceId);
				request.setProcessId(intProcessInstanceId);
			}
		}

		if (request.getProcessId() == 0) {
			List<Request> requestList = BusinessHelper.prepareRequestResponsePayload(Arrays.asList(request));

			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request. Business process failed to start.");
			return wrapper;
		}

		int id = getRequestService().createRequest(request);
		if (id > 0) {
			this.preserveHistory(request);
			wrapper = this.getRequestByID(id);
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
		} else {
			List<Request> requestList = BusinessHelper.prepareRequestResponsePayload(Arrays.asList(request));

			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request");
		}
		return wrapper;
	}

	/**
	 * Gets the request by ID.
	 *
	 * @param id   The id.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getRequestByID(int id) {
		logger.info("BusinessService: get Request by Id:" + id);
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = getRequestService().getRequestsById(id);
		if (request != null) {
			List<Request> requestList = BusinessHelper.prepareRequestResponsePayload(Arrays.asList(request));

			wrapper.setResponse(requestList.get(0));
		}
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the request tracking by ID.
	 *
	 * @param id  The id.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getRequestTrackingByID(int id) {
		logger.info("BusinessService: get Audit Request by Id:" + id);
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = getRequestService().getRequestsById(id);
		if (request != null) {
			List<RequestHistory> trackingList = getRequestService().getRequestHistoryById(id);
			if (trackingList == null || trackingList.isEmpty())
				wrapper.setResponse(null);
			else {
				// transform trackingList
				wrapper.setResponse(this.getAuditTrailWithComments(trackingList));
			}
		}
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the requests by UID.
	 *
	 * @param uid     The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getRequestsByUID(String uid) {
		logger.info("BusinessService: get Request by User:" + uid);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList = getRequestService().getRequestsByUID(uid);
		requestList = this.filterRequestList(requestList, uid);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the requests by data managers.
	 *
	 * @param uid    The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getRequestsByDataManagers(String uid) {
		logger.info("BusinessService: get Request by  Data Managers");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList = getRequestService().getRequestsDataManagers(uid);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the requests by data source managers.
	 *
	 * @param uid The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestsDataSourceManagers(String uid) {
		logger.info("BusinessService: get Request by Data Source Managers");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList = getRequestService().getRequestsDataSourceManagers(uid);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		return createResponseWrapper(wrapper);
	}
	
	/**
	 * Filter request list.
	 *
	 * @param requestList
	 *            The requestList.
	 * @param uid   The uid.
	 * @return List<Request> This returns filteredRequestList.
	 */
	private List<Request> filterRequestList(List<Request> requestList, String uid) {
		List<Request> filteredRequestList = new ArrayList<Request>();

		for (Request request : requestList) {
			if (!(!StringUtils.equalsIgnoreCase(request.getCreatedBy(), uid)
					&& StringUtils.equalsIgnoreCase(request.getStatusDescription(), "Draft"))) {
				filteredRequestList.add(request);
			}
		}
		return filteredRequestList;
	}

	/**
	 * Gets the all requests.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getAllRequests() {
		logger.info("BusinessService: get all Requests");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList = getRequestService().getAllRequests();
		logger.info("Got Requests count:" + requestList.size());
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the all requests by study approval.
	 *
	 * @param studyApprovalId The studyApprovalId.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getAllRequestsByStudyApproval(int studyApprovalId) {
		logger.info("BusinessService: getAllRequestsByStudyApproval:" + studyApprovalId);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList = getRequestService().getAllRequestsByStudyApproval(studyApprovalId);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		return createResponseWrapper(wrapper);
		
	}

	/**
	 * Gets the all requests by staus.
	 *
	 * @param status  The status.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper  getAllRequestsByStaus(String status) {
		logger.info("BusinessService: get all Requests by Status:" + status);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList = getRequestService().getAllRequestsByStaus(status);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Submit or modify.
	 *
	 * @param request    The request.
	 * status
	 */
	@Override
	public ResponseWrapper  submitOrModify(Request request) {
		logger.info("BusinessService: Sumbit/Modify Request");
		return this.createRequest(request);
	}

	/**
	 * This method is used to Persist an entitiy.
	 *
	 * @param id The id.
	 * @param changeRequestIn The changeRequestIn.
	 * @param operation The operation.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper persist(int id, Request changeRequestIn, String operation) {
		logger.info("BusinessService: Persisting: " + operation);
		ResponseWrapper wrapper = new ResponseWrapper();
		Request changeRequest = new Request();
		if (StringUtils.equalsIgnoreCase(operation, "Update") || StringUtils.equalsIgnoreCase(operation, "Submitted")) {
			changeRequest = BusinessHelper.handleRequestPayload(changeRequestIn);
		}
		Request databaseRequest = requestService.getRequestsById(id);

		if (databaseRequest == null) {
			changeRequest.setId(id);
			wrapper.setResponse(changeRequest);
			wrapper.setMessage("No such request found in database");
			wrapper.setSuccess(false);
			return wrapper;
		}

		String status = StringUtils.trim(databaseRequest.getStatusDescription());

		if (StringUtils.equalsIgnoreCase(status, "Rejected")) {
			wrapper.setResponse(changeRequest);
			wrapper.setMessage("You can't submit this request.");
			wrapper.setSuccess(false);
			return wrapper;
		}

		if (!StringUtils.equals(operation, "Update"))
			changeRequest.setStatusDescription(operation);

		if (changeRequest.getProcessId() == 0)
			changeRequest.setProcessId(databaseRequest.getProcessId());
		
		if (changeRequest.getSource() == null)
			changeRequest.setSource(databaseRequest.getSource());
		
		if (changeRequest.getStudyApproval() == null)
			changeRequest.setStudyApproval(databaseRequest.getStudyApproval());
		
		if (changeRequest.getRequestType() == null)
			changeRequest.setRequestType(databaseRequest.getRequestType());

		if (StringUtils.isBlank(changeRequest.getStatusDescription()))
			changeRequest.setStatusDescription(databaseRequest.getStatusDescription());

		if (StringUtils.isBlank(changeRequest.getTitle()))
			changeRequest.setTitle(databaseRequest.getTitle());

		if (StringUtils.isBlank(changeRequest.getDescription()))
			changeRequest.setDescription(databaseRequest.getDescription());

		changeRequest.setModifiedOn(new Date());
		changeRequest.setModifiedBy(changeRequestIn.getCreatedBy());

		changeRequest.setCreatedBy(databaseRequest.getCreatedBy());
		changeRequest.setCreatedOn(databaseRequest.getCreatedOn());
		changeRequest.setApproverId(databaseRequest.getApproverId());

		changeRequest.setId(id);
		
		if (changeRequestIn.getComments() == null) {
			changeRequest.setComments(StringUtils.EMPTY);
		} else {
			changeRequest.setComments(changeRequestIn.getComments());
		}
		boolean statusCompleteWorkflowProcess = this.completeWorkflowProcess(id, operation, changeRequest);

		if (!statusCompleteWorkflowProcess) {
			List<Request> requestList = BusinessHelper.prepareRequestResponsePayload(Arrays.asList(changeRequest));
			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage(
					operation + " failed. Error occured on the workflow. Please contact system administrator. Id#" + id);
		    return wrapper;
		}

		UUID uuid = UUID.randomUUID();
		changeRequest.setTaskId(uuid.toString());

		int submitOrModifyId = requestService.submitOrModify(changeRequest);

		if (submitOrModifyId > 0) {
			wrapper = this.getRequestByID(submitOrModifyId);
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
			preserveHistory(requestService.getRequestsById(submitOrModifyId));
			if (!StringUtils.isBlank(changeRequest.getComments())) {
				// Save Comments for declines in CommentHistory
				CommentHistory commentHistory = new CommentHistory();
				commentHistory.setComments(changeRequest.getComments());
				Request request = getRequestService().getRequestsById(submitOrModifyId);

				int statusId = getLookUpService().getStatusId(request.getStatusDescription());

				WorkflowStatus workflowStatus = new WorkflowStatus();
				workflowStatus.setId(statusId);
				commentHistory.setWorkflowStatus(workflowStatus);
				commentHistory.setRequest(request);
				commentHistory.setCreatedBy(changeRequest.getModifiedBy());
				commentHistory.setType("action");
				commentHistory.setTaskId(uuid.toString());
				getCommentHistoryService().submitOrModify(commentHistory);
			}
		} else {
			List<Request> requestList = BusinessHelper.prepareRequestResponsePayload(Arrays.asList(changeRequest));

			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request");
		}
		return wrapper;
	}

	/**
	 * Complete workflow process.
	 *
	 * @param requestId The requestId.
	 * @param operation The operation.
	 * @param changeRequest The changeRequest.
	 * @return boolean This returns true, if successful.
	 */
	private boolean completeWorkflowProcess(int requestId, String operation, Request changeRequest) {
		boolean success = true;
	
		String comments = changeRequest.getComments();
		if (comments == null)
			comments = StringUtils.EMPTY;

		List<EmailDetails> emailDetails = new ArrayList<EmailDetails>();
			
		String requestorId=null;
		String approverId=null;
			
		Request request = getRequestService().getRequestsById(requestId);
			
		RequestType requestType = getRequestService().getRequestType(request.getRequestType().getId());
		Source source = getRequestService().getSource(request.getSource().getId());
		
		Users requesterUsers = userManagementService.getUserDetailsByEmail(request.getCreatedBy());
		Users approverUsers = userManagementService.getUserDetailsByEmail(requesterUsers.getApprover());
		
		Users anyApproverUsers=null;
		if (!StringUtils.isBlank(changeRequest.getModifiedBy()))
		 anyApproverUsers = userManagementService.getUserDetailsByEmail(changeRequest.getModifiedBy());
		
	
		String status = request.getStatusDescription();
		StudyApproval studyApproval = getStudyApprovalService().getStudyApprovalsByID(request.getStudyApproval().getId());
	
		if (approverUsers == null) {
			approverId = StringUtils.EMPTY;
		} else {
			if (!StringUtils.isBlank(approverUsers.getEmailId()))
				approverId = approverUsers.getEmailId();
		}

		if (requesterUsers == null) {
			requestorId = StringUtils.EMPTY;
		} else {
			if (!StringUtils.isBlank(requesterUsers.getEmailId()))
				requestorId = requesterUsers.getEmailId();
		}
		
		
		JSONObject json = new JSONObject();
		json.put(CommonEnum.EMAIL_OPERATION.getText(), operation);
		json.put(CommonEnum.EMAIL_REQUESTOR_ID.getText(), requestorId);
		json.put(CommonEnum.EMAIL_APPROVER_DECISION.getText(), status);
		json.put(CommonEnum.EMAIL_FULFILLMENT_DECISION.getText(), "");
		json.put(CommonEnum.EMAIL_RESULT_ACCEPTANCE_DECISION.getText(), "");
		json.put(CommonEnum.EMAIL_APPROVER_ID.getText(), approverId);
		json.put(CommonEnum.EMAIL_CC_USERS.getText(),"");
		
		
				
		String pid = String.valueOf(request.getProcessId());
	
		if (StringUtils.equals(operation, CommonEnum.REQUEST_SUBMIT.getText())) {
			
					
			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			
			String emailRecipient=null;
			
			if (approverUsers == null) {
				emailRecipient = StringUtils.EMPTY;
			} else {
				if (!StringUtils.isBlank(approverUsers.getEmailId()))
					emailRecipient = approverUsers.getEmailId();
			}
			
						
			json.put(CommonEnum.EMAIL_APPROVER_DECISION.getText(), "");

			String emailSubject = "ACTION REQUIRED: Submitted Request - " + requestId + " - " + request.getTitle();
			String emailBody = "A new data request " + requestId + " has been Submitted by " + requestorId + " for review. ";
			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);

			emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			emailRecipient = requesterUsers.getEmailId();
			
			emailSubject = "Submitted Request - " + requestId + " - " + request.getTitle();
			emailBody = "Your new data request " + requestId + " has been Submitted to " + approverId + " for review. ";
			emailDet = new EmailDetails(emailSender, emailRecipient,  emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getCreatedBy(), json);
			} catch (Exception e) {
				logger.error("Unable to Claim and Complete Task for Request Submit. Reason: " + e.getMessage() + " ", e);
				success = false;
			}
		}

		else if (StringUtils.equals(operation, CommonEnum.REQUEST_RETURNED.getText())) {
			
			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String emailRecipient =  requesterUsers.getEmailId();
			

			json.put(CommonEnum.EMAIL_APPROVER_DECISION.getText(), "Return");

			String emailSubject = "Returned Request - " + requestId + " - " + request.getTitle();
			String emailBody = "Your data request " + requestId + "  has been Returned by " + approverId + System.lineSeparator();

			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;
			

			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error("Unable to Claim and Complete Task for Request Returned. Reason: " + e.getMessage() + " ", e);
				success = false;
			}
		}

		else if (StringUtils.equals(operation, CommonEnum.REQUEST_DENIED.getText())) {
			
					
			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String emailRecipient =  requesterUsers.getEmailId();
			json.put(CommonEnum.EMAIL_APPROVER_DECISION.getText(), "Reject");

			String emailSubject = "Request Denied  - " + requestId + " - " + request.getTitle();
			String emailBody = "Your data request " + requestId + "  has been Denied by " + approverId + "."
					+ System.lineSeparator();

			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;

			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, anyApproverUsers == null ? StringUtils.EMPTY : anyApproverUsers.getEmailId(), json);
			} catch (Exception e) {
				logger.error("Unable to Claim and Complete Task for Request Denied. Reason: " + e.getMessage() + " ", e);
				success = false;
			}

		}

		else if (StringUtils.equals(operation, CommonEnum.REQUEST_SENT.getText())) {
			
				
			String datasourcemanagers=genisisProperties.getDatasourcemanagers();
			
			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String emailRecipient = StringUtils.EMPTY;
			StringBuilder stringBuilder = new StringBuilder("");
			
			String result[] = StringUtils.split(datasourcemanagers, "\\,");
			for (int i = 0; i < result.length; i++) {

				if (!StringUtils.isBlank(result[i])) {
					if (i == 0) {
						emailRecipient = result[i];
					} else {
						if (i < result.length - 1)
							stringBuilder.append(result[i]).append(",");
						else
							stringBuilder.append(result[i]);
					}
				}
			}
			String cc;
			if (StringUtils.isBlank(stringBuilder.toString()))
				cc = StringUtils.EMPTY;
			else
				cc = stringBuilder.toString();
			
			json.put(CommonEnum.EMAIL_APPROVER_DECISION.getText(), "Sent");

			String emailSubject = "ACTION REQUIRED: Genisis Data Request - " + requestId + " - " + request.getTitle();
			String emailBody = "A new " + requestType.getDescription() + " data request " + requestId + " has been reviewed by "
					+ approverId + " and determined to have sufficient information for data research. "
					+ System.lineSeparator();
			emailBody = emailBody + "Request ID: " + requestId + System.lineSeparator();
			emailBody = emailBody + "Request Type: " + requestType.getDescription() + System.lineSeparator();
			emailBody = emailBody + "Data Source: " + source.getDescription() + System.lineSeparator();
			emailBody = emailBody + "Approved Study: " + studyApproval.getDescription() + System.lineSeparator();
			emailBody = emailBody + "Description: " + request.getDescription() + System.lineSeparator();
			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;

			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,cc );
			emailDetails.add(emailDet);

			emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			emailRecipient =  requesterUsers.getEmailId();
			
			emailSubject = "Submitted Request - " + requestId + " - Accepted";
			emailBody = "Your new data request " + requestId + " has been Accepted and sent to " + source.getDescription()
					+ " for research.";
			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;
			emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);
			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error(
						"Unable to Claim and Complete Task for Request Sent to VINCI. Reason: " + e.getMessage() + " ", e);
				success = false;
			}

		}

		else if (StringUtils.equals(operation, CommonEnum.REQUEST_NOT_ACCEPTED.getText())) {
		
			
			String	emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String	emailRecipient =  requesterUsers.getEmailId();
			
			
			json.put(CommonEnum.EMAIL_FULFILLMENT_DECISION.getText(), "Unfulfilled");

			String		emailSubject = "Genisis Data Request Cannot Be Fulfilled  - " + requestId + " - " + request.getTitle();
			String	emailBody = CommonEnum.EMAIL_YOUR_DATE_REQUEST.getText() + requestId + " Cannot Be Fulfilled.  " + System.lineSeparator();


			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;

			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			
			emailRecipient=	request.getModifiedBy();
			emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			
			
			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error("Unable to Claim and Complete Task for Request Not Accepted by VINCI. Reason: "
						+ e.getMessage() + " ", e);
				success = false;
			}

		}

		else if (StringUtils.equals(operation, CommonEnum.REQUEST_ACCEPTED.getText())) {
			
			
			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String emailRecipient =  requesterUsers.getEmailId();
			
			json.put(CommonEnum.EMAIL_FULFILLMENT_DECISION.getText(), "Fulfilled");
			json.put("cc_users", "");

			String emailSubject = "Genisis Data Request - " + requestId + " - " + request.getTitle();
			String emailBody = CommonEnum.EMAIL_YOUR_DATE_REQUEST.getText() + requestId + " has been Accepted for review.  " + System.lineSeparator();

			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;

			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);
			
			
			emailRecipient=	request.getModifiedBy();
			emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
				

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error("Unable to Claim and Complete Task for Request Accepted by VINCI. Reason: "
						+ e.getMessage() + " ", e);
				success = false;
			}

		}

		else if (StringUtils.equals(operation, CommonEnum.RESULTS_DELIVERED.getText())) {
			
			
		   List<RequestHistory> requestHistory=getRequestService().getRequestHistoryByRequestIdAndStatus(request.getId(), CommonEnum.REQUEST_SENT.getText());
			json.put("cc_users", "vinci.datasource@gmail.com");
					
			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String emailRecipient =  requestHistory.get(0).getModifiedBy();
			
			String emailSubject = "ACTION REQUIRED: Genisis Data Request  - " + requestId + " - " + request.getTitle();
			String emailBody = "The results from the research for  " + requestId
					+ " have been Completed and ready for your review.  " + System.lineSeparator();
			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;
			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			
			
			emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			emailRecipient =  requesterUsers.getEmailId();
			emailSubject = "Genisis Data Request   - " + requestId + " - " + request.getTitle();
			emailBody = CommonEnum.EMAIL_YOUR_DATE_REQUEST.getText() + requestId + " results were Delivered to " + approverId + " for review.  "
					+ System.lineSeparator();

			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;
			emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);

			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error(
						"Unable to Claim and Complete Task for Results Delivered. Reason: " + e.getMessage() + " ", e);
				success = false;
			}

		}

		else if (StringUtils.equals(operation, CommonEnum.RESULTS_NOT_ACCEPTED.getText())) {
			
		
			json.put(CommonEnum.EMAIL_RESULT_ACCEPTANCE_DECISION.getText(), "Reject");
			 List<RequestHistory> requestHistory=getRequestService().getRequestHistoryByRequestIdAndStatus(request.getId(), CommonEnum.RESULTS_DELIVERED.getText());
			
			
			 String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			 String emailRecipient =  requesterUsers.getEmailId();
			 String emailSubject = "Results Not  Accepted   - " + requestId + " - " + request.getTitle();
			 String emailBody = "The results from the research of  " + requestId + " have been reviewed and Not Accepted by "
					+ approverId + ".  " + System.lineSeparator();
			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;
			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			
			
			emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			emailRecipient=	requestHistory.get(0).getModifiedBy();
			
			emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);

			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error(
						"Unable to Claim and Complete Task for Results Not Accepted. Reason: " + e.getMessage() + " ", e);
				success = false;
			}

		}

		else if (StringUtils.equals(operation, CommonEnum.RESULTS_ACCEPTED.getText())) {
					
			 List<RequestHistory> requestHistory=getRequestService().getRequestHistoryByRequestIdAndStatus(request.getId(), CommonEnum.RESULTS_DELIVERED.getText());
				
			json.put(CommonEnum.EMAIL_RESULT_ACCEPTANCE_DECISION.getText(), "Accept");
			

			String emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			String emailRecipient =  requesterUsers.getEmailId();
			String emailSubject = "Results  Accepted    - " + requestId + " - " + request.getTitle();
			String emailBody = "The results from the research of  " + requestId + "  have been reviewed and Accepted by  "
					+ approverId + ".  " + System.lineSeparator();
			emailBody = emailBody + CommonEnum.EMAIL_COMMENTS.getText() + comments;
			EmailDetails emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);
			
			emailSender = CommonEnum.EMAIL_NO_REPLY_SENDER.getText();
			emailRecipient=	requestHistory.get(0).getModifiedBy();
			
			emailDet = new EmailDetails(emailSender, emailRecipient, emailSubject, emailBody,StringUtils.EMPTY);
			emailDetails.add(emailDet);

			json.put(CommonEnum.EMAIL_LIST.getText(), emailDetails);

			try {
				activitiService.claimAndCompleteHumanTask(pid, request.getApproverId(), json);
			} catch (Exception e) {
				logger.error("Unable to Claim and Complete Task for Results Accepted. Reason: " + e.getMessage() + " ", e);
				success = false;
			}
		}
		logger.info("CompleteWorkflowProcess:    Success status: " + success);
		return success;
	}

	/**
	 * Preserve history.
	 *
	 * @param request
	 *            The request.
	 */
	private void preserveHistory(Request request) {
	
		

		RequestHistory requestHistory = new RequestHistory(0, request, request.getProcessId(), request.getTitle(),
				request.getDescription(),  
				request.getStatusDescription(), request.getRequestType().getId(), request.getSource().getId(),
		  	    request.getStudyApproval().getId(), request.getCreatedOn(),
				request.getModifiedOn(), request.getCreatedBy(), request.getModifiedBy(), request.getApproverId(),
				request.getTaskId());

		int returnId = getRequestService().submitOrModify(requestHistory);
		if (returnId > 0) {

			logger.info(requestHistory.getRequest().getId() + " with status " + requestHistory.getStatusDescription()
					+ " added to the history successfully!!");
		} else {
			logger.warn(requestHistory.getRequest().getId() + " with status " + requestHistory.getStatusDescription()
					+ " could not be added to the history.");
		}

	}

	/**
	 * Creates the comment history.
	 *
	 * @param commentHistory
	 *            The commentHistory.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  createCommentHistory(CommentHistory commentHistory) {
		logger.info("BusinessService:  createCommentHistory");
		ResponseWrapper wrapper = new ResponseWrapper();
		int statusId = getLookUpService().getStatusId(commentHistory.getStatus());
		WorkflowStatus workflowStatus = new WorkflowStatus();
		workflowStatus.setId(statusId);

		Request request = getRequestService().getRequestsById(commentHistory.getRequestId());
		commentHistory.setWorkflowStatus(workflowStatus);
		commentHistory.setRequest(request);

		commentHistory.setTaskId(request.getTaskId());

		int id = getCommentHistoryService().submitOrModify(commentHistory);

		if (id > 0) {
		
			CommentHistory returnCommentHistoryResp = getCommentHistoryService().getCommentHistoryByCommentId(id);
			returnCommentHistoryResp.setRequestId(commentHistory.getRequestId());
			returnCommentHistoryResp.setStatus(commentHistory.getStatus());
			wrapper.setResponse(returnCommentHistoryResp);

			wrapper.setSuccess(true);
			wrapper.setMessage(null);
		} else {
			List<CommentHistory> commentHistoryList = BusinessHelper
					.prepareCommentHistoryResponsePayload(Arrays.asList(commentHistory));
			wrapper.setResponse(commentHistoryList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Comment History");
		}
		return wrapper;
	}

	/**
	 * Gets the audit trail with comments.
	 *
	 * @param trackingList The trackingList.
	 * @return List<gov.va.genisis2.vo.RequestHistory> This returns returnRequestHistory.
	 */
	private List<gov.va.genisis2.vo.RequestHistory> getAuditTrailWithComments(List<RequestHistory> trackingList) {
		List<gov.va.genisis2.vo.RequestHistory> returnRequestHistory = new ArrayList<gov.va.genisis2.vo.RequestHistory>();
		for (RequestHistory requestHistory : trackingList) {

			List<CommentHistory> commentHistory = getCommentHistoryService().getCommentHistoryByRequestIdStatusId(
					requestHistory.getRequest().getId(), requestHistory.getTaskId());

			gov.va.genisis2.vo.RequestHistory element = new gov.va.genisis2.vo.RequestHistory();
			element.setCommentHistory(commentHistory);
			element.setCreatedBy(requestHistory.getCreatedBy());
			element.setCreatedOn(requestHistory.getCreatedOn());
			element.setStatusDescription(requestHistory.getStatusDescription());
			element.setTitle(requestHistory.getTitle());
			element.setDescription(requestHistory.getDescription());
			element.setModifiedBy(requestHistory.getModifiedBy());
			element.setModifiedOn(requestHistory.getModifiedOn());

			returnRequestHistory.add(element);

		}
		return returnRequestHistory;
	}

	/**
	 * Perform table copy transfer.
	 *
	 * @param jsonRequest          The jsonRequest.
	 * @param requestId           The requestId.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  performTableCopyTransfer(CopyTableSource jsonRequest, int requestId) {
		logger.info("BusinessService:  performTableCopyTransfer");
		
		ResponseWrapper wrapper = new ResponseWrapper();
		wrapper.setSuccess(true);
		
	
		// Get required properties
		String tcSourceDataBaseServerName=copyTableSource.getTcSourceDataBaseServerName();
		String tcSourceDataBaseName=copyTableSource.getTcSourceDataBaseName();
		String sourceSchemaName=jsonRequest.getSourceSchemaName();
		String tcDestinationDataBaseServerName=copyTableSource.getTcDestinationDataBaseServerName();
		String tcDestinationDataBaseUserName=copyTableSource.getTcDestinationDataBaseUserName();
		String tcDestinationDataBaseUserPassword=copyTableSource.getTcDestinationDataBaseUserPassword();
		String tcDestinationServerDataBaseName=copyTableSource.getTcDestinationServerDataBaseName();
		String destinationSchemaName=jsonRequest.getDestinationSchemaName();
		String tcManagementDataBaseServerName=copyTableSource.getTcManagementDataBaseServerName();
		String tcManagementDataBaseUserName=copyTableSource.getTcManagementDataBaseUserName();
		String tcManagementDataBaseUserPassword=copyTableSource.getTcManagementDataBaseUserPassword();
		String tcManagementDataBaseSchema=copyTableSource.getTcManagementDataBaseSchema();
		String copyTable = jsonRequest.getTableName();
		String copyTableScript = copyTableSource.getCmdFilePath();
		String tcManagementDataBaseName=copyTableSource.getTcManagementDataBaseName();
		String tcTemporaryDataBaseName=copyTableSource.getTcTemporaryDataBaseName();
		
		// Check for all properties are present
	           if (StringUtils.isBlank(tcSourceDataBaseServerName) || StringUtils.isBlank(tcSourceDataBaseName)
	      		|| StringUtils.isBlank(sourceSchemaName) || StringUtils.isBlank(tcDestinationDataBaseServerName)
				|| StringUtils.isBlank(tcDestinationDataBaseUserName) || StringUtils.isBlank(tcDestinationDataBaseUserPassword)
				|| StringUtils.isBlank(tcDestinationServerDataBaseName) || StringUtils.isBlank(destinationSchemaName)
				|| StringUtils.isBlank(tcManagementDataBaseServerName) || StringUtils.isBlank(tcManagementDataBaseUserName)
				|| StringUtils.isBlank(tcManagementDataBaseUserPassword) || StringUtils.isBlank(tcManagementDataBaseSchema)
				|| StringUtils.isBlank(tcManagementDataBaseName) || StringUtils.isBlank(tcTemporaryDataBaseName)
				|| StringUtils.isBlank(copyTable) || StringUtils.isBlank(copyTableScript)
				) {

			logger.error("One or more properties in genisis2.properties file seems blank");
			wrapper.setResponse(jsonRequest);
			wrapper.setMessage("One or more properties in genisis2.properties file seems blank");
			wrapper.setSuccess(false);
			return wrapper;
		}

		File file = new File(copyTableScript);

		// Check shell script available on Linux server
		if (!file.isFile()) {
			logger.error("The file " + copyTableScript + " does not exist");
			wrapper.setResponse(jsonRequest);
			wrapper.setMessage("Copy table failed. " + copyTableScript + " does not exist");
			wrapper.setSuccess(false);
			return wrapper;
		}
		
	/*	IP          GENISISDATASOURCE dflt IP          data_ops genisis20!7 GENISISDATADEST dbo IP          data_ops genisis20!7 
	  dbo vinciTable1,vinciTable2 	*/
		
		StringBuilder builder = new StringBuilder();
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(copyTableScript)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcSourceDataBaseServerName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcSourceDataBaseName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(sourceSchemaName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcDestinationDataBaseServerName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcDestinationDataBaseUserName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcDestinationDataBaseUserPassword)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcDestinationServerDataBaseName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(destinationSchemaName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcManagementDataBaseServerName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcManagementDataBaseUserName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcManagementDataBaseUserPassword)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcManagementDataBaseName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcTemporaryDataBaseName)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(tcManagementDataBaseSchema)).append('\'');
		builder.append(CommonEnum.SPACE.getText()).append('\'');
		builder.append(StringUtils.trim(copyTable)).append('\'');
		logger.info("Arguments being passed to DataOps : " +builder.toString());
		
		 String[] command = new String[16];
		 command[0] = copyTableScript;
		 command[1] = tcSourceDataBaseServerName;
		 command[2] = tcSourceDataBaseName;
		 command[3] = sourceSchemaName;
		 command[4] = tcDestinationDataBaseServerName;
		 command[5] = tcDestinationDataBaseUserName;
		 command[6] = tcDestinationDataBaseUserPassword;
		 command[7] = tcDestinationServerDataBaseName;
		 command[8] =  destinationSchemaName;
		 command[9] = tcManagementDataBaseServerName;
		 command[10] = tcManagementDataBaseUserName;
		 command[11] = tcManagementDataBaseUserPassword;
		 command[12] = tcManagementDataBaseName;
		 command[13] = tcTemporaryDataBaseName;
		 command[14] = tcManagementDataBaseSchema;
		 command[15] = copyTable;

		Process process = null;
		BufferedReader reader = null;
		String line = StringUtils.EMPTY;
		logger.info( "Table/Copy arguments: " + Arrays.toString(command));
		

		try {
			Runtime runtime = Runtime.getRuntime();
			process = runtime.exec(command);
			process.waitFor();
			reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
			while ((line = reader.readLine()) != null) {
				wrapper.setResponse(line);
			}
			
			logger.info( "DataOps  Response: " + wrapper.getResponse());
		//} catch (IOException | InterruptedException  ex) {
		 } catch (Exception ex) {
			logger.error("Error occured while copying table(s). Reason: " + ex.getMessage() + " ", ex);
			wrapper.setResponse(jsonRequest);
			wrapper.setMessage("Copy Table failed. Please contact system administrator.");
			wrapper.setSuccess(false);
		} finally {
			try {
				if (process != null && process.isAlive())
					process.destroy();
				if (reader != null)
					reader.close();
			} catch (IOException ex) {
				logger.error("Error occured while closing the resources. Reason: " + ex.getMessage(), ex);
			}
		}
		return wrapper;
	}

	/**
	 * Gets the copy table domains.
	 *
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper  getCopyTableDomains() {
		logger.info("BusinessService: Copy Table Domains");
		ResponseWrapper wrapper = new ResponseWrapper();
		wrapper.setMessage("");
		wrapper.setSuccess(true);
			
		CopyTableDomains tableDomains = new CopyTableDomains(copyTableDomains.getSourceDBName(),
				copyTableDomains.getDestinationDBName(), copyTableDomains.getSourceDBServerName(),
				copyTableDomains.getDestinationDBServerName(), copyTableDomains.getsUsername(),
				copyTableDomains.getdUsername());
		wrapper.setResponse(tableDomains);
		return wrapper;
	}


	
	

	

	/**
	 * @return the genisisProperties
	 */
	public GenisisProperties getGenisisProperties() {
		return genisisProperties;
	}

	/**
	 * @param genisisProperties the genisisProperties to set
	 */
	@Autowired
	public void setGenisisProperties(GenisisProperties genisisProperties) {
		this.genisisProperties = genisisProperties;
	}

	/**
	 * @return the copyTableSource
	 */
	public CopyTableSource getCopyTableSource() {
		return copyTableSource;
	}

	/**
	 * @param copyTableSource the copyTableSource to set
	 */
	public void setCopyTableSource(CopyTableSource copyTableSource) {
		this.copyTableSource = copyTableSource;
	}

	/**
	 * @param copyTableDomains the copyTableDomains to set
	 */
	@Autowired
	public void setCopyTableDomains(CopyTableDomains copyTableDomains) {
		this.copyTableDomains = copyTableDomains;
	}

	/**
	 * Gets the copy table source.
	 *
	 * @param copyTableSource the copy table source
	 *
	 */
	@Autowired
	public void getCopyTableSource(CopyTableSource copyTableSource) {
		this.copyTableSource = copyTableSource;
	}

	/**
	 * Gets the look up service.
	 *
	 * @return LookUpService This returns lookUpService.
	 */
	public LookUpService getLookUpService() {
		return lookUpService;
	}

	/**
	 * Sets the look up service.
	 *
	 * @param lookUpService The lookUpService.
	 */
	@Autowired
	public void setLookUpService(LookUpService lookUpService) {
		this.lookUpService = lookUpService;
	}

	/**
	 * Gets the activiti service.
	 *
	 * @return ActivitiService This returns activiti service.
	 */
	public ActivitiService getActivitiService() {
		return activitiService;
	}

	/**
	 * Sets the activiti service.
	 *
	 * @param activitiService The activitiService.
	 */
	@Autowired
	public void setActivitiService(ActivitiService activitiService) {
		this.activitiService = activitiService;
	}

	/**
	 * Gets the request service.
	 *
	 * @return RequestService This returns requestService.
	 */
	public RequestService getRequestService() {
		return requestService;
	}

	/**
	 * Sets the request service.
	 *
	 * @param requestService The requestService.
	 */
	@Autowired
	public void setRequestService(RequestService requestService) {
		this.requestService = requestService;
	}

	/**
	 * Gets the study approval service.
	 *
	 * @return StudyApprovalService This returns studyApprovalService.
	 */
	public StudyApprovalService getStudyApprovalService() {
		return studyApprovalService;
	}

	/**
	 * Sets the study approval service.
	 *
	 * @param studyApprovalService The studyApprovalService.
	 */
	@Autowired
	public void setStudyApprovalService(StudyApprovalService studyApprovalService) {
		this.studyApprovalService = studyApprovalService;
	}

	/**
	 * Gets the comment history service.
	 *
	 * @return CommentHistoryService This returns commentHistoryService.
	 */
	public CommentHistoryService getCommentHistoryService() {
		return commentHistoryService;
	}

	/**
	 * Sets the comment history service.
	 *
	 * @param commentHistoryService The commentHistoryService.
	 */
	@Autowired
	public void setCommentHistoryService(CommentHistoryService commentHistoryService) {
		this.commentHistoryService = commentHistoryService;
	}

	/**
	 * Get the user management service.
	 *
	 * @return UserManagementService This returns userManagementService.
	 */
	public UserManagementService getUserManagementService() {
		return userManagementService;
	}

	/**
	 * Set the user management service.
	 *
	 * @param userManagementService The userManagementService.
	 */
	@Autowired
	public void setUserManagementService(UserManagementService userManagementService) {
		this.userManagementService = userManagementService;
	}

	
}
